"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.emailQueue = void 0;
exports.sendEmail = sendEmail;
exports.sendChatEmail = sendChatEmail;
exports.sendFiatTransactionEmail = sendFiatTransactionEmail;
exports.sendBinaryOrderEmail = sendBinaryOrderEmail;
exports.sendWalletBalanceUpdateEmail = sendWalletBalanceUpdateEmail;
exports.sendTransactionStatusUpdateEmail = sendTransactionStatusUpdateEmail;
exports.sendAuthorStatusUpdateEmail = sendAuthorStatusUpdateEmail;
exports.sendOutgoingTransferEmail = sendOutgoingTransferEmail;
exports.sendIncomingTransferEmail = sendIncomingTransferEmail;
exports.sendSpotWalletWithdrawalConfirmationEmail = sendSpotWalletWithdrawalConfirmationEmail;
exports.sendSpotWalletDepositConfirmationEmail = sendSpotWalletDepositConfirmationEmail;
exports.sendAiInvestmentEmail = sendAiInvestmentEmail;
exports.sendInvestmentEmail = sendInvestmentEmail;
exports.sendIcoContributionEmail = sendIcoContributionEmail;
exports.sendStakingInitiationEmail = sendStakingInitiationEmail;
exports.sendStakingRewardEmail = sendStakingRewardEmail;
exports.sendOrderConfirmationEmail = sendOrderConfirmationEmail;
exports.sendEmailToTargetWithTemplate = sendEmailToTargetWithTemplate;
exports.sendKycEmail = sendKycEmail;
exports.sendForexTransactionEmail = sendForexTransactionEmail;
const bull_1 = __importDefault(require("bull"));
const mailer_1 = require("./mailer");
const date_fns_1 = require("date-fns");
const db_1 = require("@b/db");
const APP_EMAILER = process.env.APP_EMAILER || "nodemailer-service";
exports.emailQueue = new bull_1.default("emailQueue", {
    redis: {
        host: "127.0.0.1",
        port: 6379,
    },
});
exports.emailQueue.process(async (job) => {
    const { emailData, emailType } = job.data;
    try {
        await sendEmail(emailData, emailType);
        console.log("Email sent successfully");
    }
    catch (error) {
        console.error("Failed to send email:", error);
        // Optionally: Re-queue or handle the job based on the error
        throw error;
    }
});
async function sendEmail(specificVariables, templateName) {
    let processedTemplate;
    let processedSubject;
    try {
        const result = await (0, mailer_1.fetchAndProcessEmailTemplate)(specificVariables, templateName);
        processedTemplate = result.processedTemplate;
        processedSubject = result.processedSubject;
    }
    catch (error) {
        console.error("Error processing email template:", error);
        throw error;
    }
    let finalEmailHtml;
    try {
        finalEmailHtml = await (0, mailer_1.prepareEmailTemplate)(processedTemplate, processedSubject);
    }
    catch (error) {
        console.error("Error preparing email template:", error);
        throw error;
    }
    const options = {
        to: specificVariables["TO"],
        subject: processedSubject,
        html: finalEmailHtml,
    };
    const emailer = APP_EMAILER;
    try {
        await (0, mailer_1.sendEmailWithProvider)(emailer, options);
    }
    catch (error) {
        console.error("Error sending email with provider:", error);
        throw error;
    }
}
async function sendChatEmail(sender, receiver, chat, message, emailType) {
    const emailData = {
        TO: receiver.email,
        SENDER_NAME: sender.firstName,
        RECEIVER_NAME: receiver.firstName,
        MESSAGE: message.text,
        TICKET_ID: chat.id,
    };
    await exports.emailQueue.add({
        emailData,
        emailType,
    });
}
async function sendFiatTransactionEmail(user, transaction, currency, newBalance) {
    // Define the type of email template to use, which matches the SQL record
    const emailType = "FiatWalletTransaction";
    // Prepare the email data
    const emailData = {
        TO: user.email,
        FIRSTNAME: user.firstName,
        TRANSACTION_TYPE: transaction.type,
        TRANSACTION_ID: transaction.id,
        AMOUNT: transaction.amount,
        CURRENCY: currency,
        TRANSACTION_STATUS: transaction.status,
        NEW_BALANCE: newBalance,
        DESCRIPTION: transaction.description || "N/A",
    };
    // Send the email
    await exports.emailQueue.add({ emailData, emailType });
}
async function sendBinaryOrderEmail(user, order) {
    // Define the type of email template to use, which matches the SQL record
    const emailType = "BinaryOrderResult";
    let profit = 0;
    let sign;
    switch (order.status) {
        case "WIN":
            profit = order.amount + order.amount * (order.profit / 100);
            sign = "+";
            break;
        case "LOSS":
            profit = order.amount;
            sign = "-";
            break;
        case "DRAW":
            profit = 0;
            sign = "";
            break;
    }
    const currency = order.symbol.split("/")[1];
    // Prepare the email data
    const emailData = {
        TO: user.email,
        FIRSTNAME: user.firstName,
        ORDER_ID: order.id,
        RESULT: order.status,
        MARKET: order.symbol,
        CURRENCY: currency,
        AMOUNT: order.amount,
        PROFIT: `${sign}${profit}`,
        ENTRY_PRICE: order.price,
        CLOSE_PRICE: order.closePrice,
        SIDE: order.side,
    };
    // Send the email
    await exports.emailQueue.add({ emailData, emailType });
}
async function sendWalletBalanceUpdateEmail(user, wallet, action, amount, newBalance) {
    // Define the type of email template to use, which matches the SQL record
    const emailType = "WalletBalanceUpdate";
    // Prepare the email data
    const emailData = {
        TO: user.email,
        FIRSTNAME: user.firstName,
        ACTION: action,
        AMOUNT: amount,
        CURRENCY: wallet.currency,
        NEW_BALANCE: newBalance,
    };
    // Send the email
    await exports.emailQueue.add({ emailData, emailType });
}
async function sendTransactionStatusUpdateEmail(user, transaction, wallet, newBalance, note) {
    // Define the type of email template to use, which matches the SQL record
    const emailType = "TransactionStatusUpdate";
    // Prepare the email data
    const emailData = {
        TO: user.email,
        FIRSTNAME: user.firstName,
        TRANSACTION_TYPE: transaction.type,
        TRANSACTION_ID: transaction.id,
        TRANSACTION_STATUS: transaction.status,
        AMOUNT: transaction.amount,
        CURRENCY: wallet.currency,
        NEW_BALANCE: newBalance,
        NOTE: note || "N/A",
    };
    // Send the email
    await exports.emailQueue.add({ emailData, emailType });
}
async function sendAuthorStatusUpdateEmail(user, author) {
    // Define the type of email template to use, which matches the SQL record
    const emailType = "AuthorStatusUpdate";
    // Prepare the email data
    const emailData = {
        TO: user.email,
        FIRSTNAME: user.firstName,
        AUTHOR_STATUS: author.status,
        APPLICATION_ID: author.id,
    };
    // Send the email
    await exports.emailQueue.add({ emailData, emailType });
}
async function sendOutgoingTransferEmail(user, toUser, wallet, amount, transactionId) {
    const emailType = "OutgoingWalletTransfer";
    const emailData = {
        TO: user.email,
        FIRSTNAME: user.firstName,
        AMOUNT: amount,
        CURRENCY: wallet.currency,
        NEW_BALANCE: wallet.balance,
        TRANSACTION_ID: transactionId,
        RECIPIENT_NAME: `${toUser.firstName} ${toUser.lastName}`,
    };
    await exports.emailQueue.add({ emailData, emailType });
}
async function sendIncomingTransferEmail(user, fromUser, wallet, amount, transactionId) {
    const emailType = "IncomingWalletTransfer";
    const emailData = {
        TO: user.email,
        FIRSTNAME: user.firstName,
        AMOUNT: amount,
        CURRENCY: wallet.currency,
        NEW_BALANCE: wallet.balance,
        TRANSACTION_ID: transactionId,
        SENDER_NAME: `${fromUser.firstName} ${fromUser.lastName}`,
    };
    await exports.emailQueue.add({ emailData, emailType });
}
async function sendSpotWalletWithdrawalConfirmationEmail(user, transaction, wallet) {
    // Define the type of email template to use, which matches the SQL record
    const emailType = "SpotWalletWithdrawalConfirmation";
    // Prepare the email data
    const emailData = {
        TO: user.email,
        FIRSTNAME: user.firstName,
        AMOUNT: transaction.amount,
        CURRENCY: wallet.currency,
        ADDRESS: transaction.metadata.address,
        FEE: transaction.fee,
        CHAIN: transaction.metadata.chain,
        MEMO: transaction.metadata.memo || "N/A",
        STATUS: transaction.status,
    };
    // Send the email
    await exports.emailQueue.add({ emailData, emailType });
}
async function sendSpotWalletDepositConfirmationEmail(user, transaction, wallet, chain) {
    // Define the type of email template to use, which should match the SQL record
    const emailType = "SpotWalletDepositConfirmation";
    // Prepare the email data
    const emailData = {
        TO: user.email,
        FIRSTNAME: user.firstName,
        TRANSACTION_ID: transaction.referenceId,
        AMOUNT: transaction.amount,
        CURRENCY: wallet.currency,
        CHAIN: chain,
        FEE: transaction.fee,
    };
    // Send the email
    await exports.emailQueue.add({ emailData, emailType });
}
async function sendAiInvestmentEmail(user, plan, duration, investment, emailType) {
    const resultSign = investment.result === "WIN" ? "+" : investment.result === "LOSS" ? "-" : "";
    const emailData = {
        TO: user.email,
        FIRSTNAME: user.firstName,
        PLAN_NAME: plan.title,
        AMOUNT: investment.amount.toString(),
        CURRENCY: investment.symbol.split("/")[1],
        DURATION: duration.duration.toString(),
        TIMEFRAME: duration.timeframe,
        STATUS: investment.status,
        PROFIT: investment.profit !== undefined
            ? `${resultSign}${investment.profit}`
            : "N/A",
    };
    await exports.emailQueue.add({ emailData, emailType });
}
async function sendInvestmentEmail(user, plan, duration, investment, emailType) {
    const resultSign = investment.result === "WIN" ? "+" : investment.result === "LOSS" ? "-" : "";
    const emailData = {
        TO: user.email,
        FIRSTNAME: user.firstName,
        PLAN_NAME: plan.title,
        AMOUNT: investment.amount.toString(),
        DURATION: duration.duration.toString(),
        TIMEFRAME: duration.timeframe,
        STATUS: investment.status,
        PROFIT: `${resultSign}${investment.profit}` || "N/A",
    };
    await exports.emailQueue.add({ emailData, emailType });
}
async function sendIcoContributionEmail(user, contribution, token, phase, emailType, transactionId) {
    const contributionDate = new Date(contribution.createdAt).toLocaleDateString("en-US", {
        year: "numeric",
        month: "long",
        day: "numeric",
        hour: "2-digit",
        minute: "2-digit",
    });
    // Common email data
    const emailData = {
        TO: user.email,
        FIRSTNAME: user.firstName,
        TOKEN_NAME: token.name,
        PHASE_NAME: phase.name,
        AMOUNT: contribution.amount.toString(),
        CURRENCY: token.purchaseCurrency,
        DATE: contributionDate,
    };
    // Customize email data based on the type
    if (emailType === "IcoContributionPaid") {
        emailData["TRANSACTION_ID"] = transactionId || "N/A";
    }
    else if (emailType === "IcoNewContribution") {
        emailData["CONTRIBUTION_STATUS"] = contribution.status;
    }
    await exports.emailQueue.add({ emailData, emailType });
}
// Function to send an email when a user initiates a stake
async function sendStakingInitiationEmail(user, stake, pool, reward) {
    const stakeDate = new Date(stake.stakeDate).toLocaleDateString("en-US", {
        year: "numeric",
        month: "long",
        day: "numeric",
        hour: "2-digit",
        minute: "2-digit",
    });
    const releaseDate = new Date(stake.releaseDate).toLocaleDateString("en-US", {
        year: "numeric",
        month: "long",
        day: "numeric",
        hour: "2-digit",
        minute: "2-digit",
    });
    const emailData = {
        TO: user.email,
        FIRSTNAME: user.firstName,
        TOKEN_NAME: pool.name,
        STAKE_AMOUNT: stake.amount.toString(),
        TOKEN_SYMBOL: pool.currency,
        STAKE_DATE: stakeDate,
        RELEASE_DATE: releaseDate,
        EXPECTED_REWARD: reward,
    };
    await exports.emailQueue.add({
        emailData,
        emailType: "StakingInitiationConfirmation",
    });
}
async function sendStakingRewardEmail(user, stake, pool, reward) {
    const distributionDate = (0, date_fns_1.format)(new Date(stake.releaseDate), "MMMM do, yyyy 'at' hh:mm a");
    const emailData = {
        TO: user.email,
        FIRSTNAME: user.firstName,
        TOKEN_NAME: pool.name,
        REWARD_AMOUNT: reward.toString(),
        TOKEN_SYMBOL: pool.currency,
        DISTRIBUTION_DATE: distributionDate,
    };
    await exports.emailQueue.add({ emailData, emailType: "StakingRewardDistribution" });
}
async function sendOrderConfirmationEmail(user, order, product) {
    var _a, _b, _c;
    const orderDate = new Date(order.createdAt).toLocaleDateString("en-US", {
        year: "numeric",
        month: "long",
        day: "numeric",
    });
    // Get the order with items to calculate proper totals
    const fullOrder = await db_1.models.ecommerceOrder.findByPk(order.id, {
        include: [
            {
                model: db_1.models.ecommerceOrderItem,
                as: "orderItems",
                include: [
                    {
                        model: db_1.models.ecommerceProduct,
                        as: "product",
                    },
                ],
            },
        ],
    });
    // Calculate order totals
    const subtotal = ((_a = fullOrder === null || fullOrder === void 0 ? void 0 : fullOrder.orderItems) === null || _a === void 0 ? void 0 : _a.reduce((total, item) => {
        return total + (item.product.price * item.quantity);
    }, 0)) || product.price;
    // Get system settings for shipping and tax
    const systemSettings = await db_1.models.settings.findAll();
    const settings = systemSettings.reduce((acc, setting) => {
        acc[setting.key] = setting.value;
        return acc;
    }, {});
    // Calculate shipping and tax
    let shippingCost = 0;
    if (product.type === "PHYSICAL" && settings.ecommerceShippingEnabled === "true") {
        shippingCost = parseFloat(settings.ecommerceDefaultShippingCost || "0");
    }
    let taxAmount = 0;
    if (settings.ecommerceTaxEnabled === "true") {
        const taxRate = parseFloat(settings.ecommerceDefaultTaxRate || "0") / 100;
        taxAmount = subtotal * taxRate;
    }
    const orderTotal = subtotal + shippingCost + taxAmount;
    const emailData = {
        TO: user.email,
        CUSTOMER_NAME: user.firstName,
        ORDER_NUMBER: order.id,
        ORDER_DATE: orderDate,
        PRODUCT_NAME: product.name,
        QUANTITY: ((_c = (_b = fullOrder === null || fullOrder === void 0 ? void 0 : fullOrder.orderItems) === null || _b === void 0 ? void 0 : _b[0]) === null || _c === void 0 ? void 0 : _c.quantity) || 1,
        PRODUCT_PRICE: product.price.toString(),
        PRODUCT_CURRENCY: product.currency,
        SUBTOTAL: subtotal.toFixed(2),
        SHIPPING_COST: shippingCost.toFixed(2),
        TAX_AMOUNT: taxAmount.toFixed(2),
        ORDER_TOTAL: orderTotal.toFixed(2),
        ORDER_STATUS: order.status,
        PRODUCT_TYPE: product.type,
    };
    await exports.emailQueue.add({ emailData, emailType: "OrderConfirmation" });
}
/**
 * Send an email to a specific target with a provided HTML template.
 *
 * @param {string} to - The email address of the target recipient.
 * @param {string} subject - The subject of the email.
 * @param {string} html - The HTML content to be sent.
 * @returns {Promise<void>} - The result of the email sending operation.
 */
async function sendEmailToTargetWithTemplate(to, subject, html) {
    // Options for the email.
    const options = {
        to,
        subject,
        html,
    };
    // Select the email provider.
    const emailer = APP_EMAILER;
    await (0, mailer_1.sendEmailWithProvider)(emailer, options);
}
async function sendKycEmail(user, kyc, type) {
    // For submission emails, use CREATED_AT; otherwise (updates) use UPDATED_AT.
    const timestampLabel = type === "KycSubmission" ? "CREATED_AT" : "UPDATED_AT";
    const timestampDate = type === "KycSubmission"
        ? new Date(kyc.createdAt).toISOString()
        : new Date(kyc.updatedAt).toISOString();
    // Prepare email data using the correct fields.
    // Note: kyc.level now holds the level number (or "N/A" if not found).
    const emailData = {
        TO: user.email,
        FIRSTNAME: user.firstName,
        [timestampLabel]: timestampDate,
        LEVEL: kyc.level,
        STATUS: kyc.status,
    };
    // For a rejected application, include the rejection message from adminNotes.
    if (type === "KycRejected" && kyc.adminNotes) {
        emailData["MESSAGE"] = kyc.adminNotes;
    }
    // Add the email to the queue using your emailQueue system.
    await exports.emailQueue.add({ emailData, emailType: type });
}
async function sendForexTransactionEmail(user, transaction, account, currency, transactionType) {
    const emailData = {
        TO: user.email,
        FIRSTNAME: user.firstName,
        ACCOUNT_ID: account.accountId,
        TRANSACTION_ID: transaction.id,
        AMOUNT: transaction.amount.toString(),
        CURRENCY: currency,
        STATUS: transaction.status,
    };
    let emailType = "";
    if (transactionType === "FOREX_DEPOSIT") {
        emailType = "ForexDepositConfirmation";
    }
    else if (transactionType === "FOREX_WITHDRAW") {
        emailType = "ForexWithdrawalConfirmation";
    }
    await exports.emailQueue.add({ emailData, emailType });
}
