"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.mapChainNameToChainId = mapChainNameToChainId;
const ccxt = __importStar(require("ccxt"));
const system_1 = require("./system");
const db_1 = require("@b/db");
const logger_1 = require("@b/utils/logger");
const utils_1 = require("@b/api/exchange/utils");
class ExchangeManager {
    constructor() {
        this.exchangeCache = new Map();
        this.initializationPromises = new Map();
        this.provider = null;
        this.exchange = null;
        this.exchangeProvider = null;
        this.lastAttemptTime = null;
        this.attemptCount = 0;
        this.isInitializing = false;
        this.initializationQueue = [];
    }
    async fetchActiveProvider() {
        try {
            const provider = await db_1.models.exchange.findOne({
                where: {
                    status: true,
                },
            });
            if (!provider) {
                return null;
            }
            return provider.name;
        }
        catch (error) {
            (0, logger_1.logError)("exchange", error, __filename);
            return null;
        }
    }
    async initializeExchange(provider, retries = 3) {
        if (await (0, utils_1.handleBanStatus)(await (0, utils_1.loadBanStatus)())) {
            return null;
        }
        if (this.exchangeCache.has(provider)) {
            return this.exchangeCache.get(provider);
        }
        const now = Date.now();
        if (this.attemptCount >= 3 &&
            this.lastAttemptTime &&
            now - this.lastAttemptTime < 30 * 60 * 1000) {
            return null;
        }
        const apiKey = process.env[`APP_${provider.toUpperCase()}_API_KEY`];
        const apiSecret = process.env[`APP_${provider.toUpperCase()}_API_SECRET`];
        const apiPassphrase = process.env[`APP_${provider.toUpperCase()}_API_PASSPHRASE`];
        if (!apiKey || !apiSecret || apiKey === "" || apiSecret === "") {
            (0, logger_1.logError)("exchange", new Error(`API credentials for ${provider} are missing.`), __filename);
            this.attemptCount += 1;
            this.lastAttemptTime = now;
            return null;
        }
        try {
            let exchange = new ccxt.pro[provider]({
                apiKey,
                secret: apiSecret,
                password: apiPassphrase,
            });
            const credentialsValid = await exchange.checkRequiredCredentials();
            if (!credentialsValid) {
                (0, logger_1.logError)("exchange", new Error(`API credentials for ${provider} are invalid.`), __filename);
                await exchange.close();
                exchange = new ccxt.pro[provider]();
            }
            try {
                await exchange.loadMarkets();
            }
            catch (error) {
                if (this.isRateLimitError(error)) {
                    await this.handleRateLimitError(provider);
                    return this.initializeExchange(provider, retries);
                }
                else {
                    (0, logger_1.logError)("exchange", new Error(`Failed to load markets: ${error.message}`), __filename);
                    await exchange.close();
                    exchange = new ccxt.pro[provider]();
                }
            }
            this.exchangeCache.set(provider, exchange);
            this.attemptCount = 0;
            this.lastAttemptTime = null;
            return exchange;
        }
        catch (error) {
            (0, logger_1.logError)("exchange", error, __filename);
            this.attemptCount += 1;
            this.lastAttemptTime = now;
            if (retries > 0 &&
                (this.attemptCount < 3 || now - this.lastAttemptTime >= 30 * 60 * 1000)) {
                await (0, system_1.sleep)(5000);
                return this.initializeExchange(provider, retries - 1);
            }
            return null;
        }
    }
    isRateLimitError(error) {
        return error instanceof ccxt.RateLimitExceeded || error.code === -1003;
    }
    async handleRateLimitError(provider) {
        const banTime = Date.now() + 60000; // Ban for 1 minute
        await (0, utils_1.saveBanStatus)(banTime);
        await (0, system_1.sleep)(60000); // Wait for 1 minute
    }
    async startExchange() {
        if (await (0, utils_1.handleBanStatus)(await (0, utils_1.loadBanStatus)())) {
            return null;
        }
        if (this.exchange) {
            return this.exchange;
        }
        // Handle concurrent initialization
        if (this.isInitializing) {
            return new Promise((resolve, reject) => {
                this.initializationQueue.push({ resolve, reject });
            });
        }
        this.isInitializing = true;
        try {
            this.provider = this.provider || (await this.fetchActiveProvider());
            if (!this.provider) {
                this.resolveQueue(null);
                return null;
            }
            // Check if exchange is already cached
            if (this.exchangeCache.has(this.provider)) {
                this.exchange = this.exchangeCache.get(this.provider);
                this.resolveQueue(this.exchange);
                return this.exchange;
            }
            // Initialize exchange
            this.exchange = await this.initializeExchange(this.provider);
            this.resolveQueue(this.exchange);
            return this.exchange;
        }
        catch (error) {
            this.rejectQueue(error);
            throw error;
        }
        finally {
            this.isInitializing = false;
        }
    }
    resolveQueue(result) {
        while (this.initializationQueue.length > 0) {
            const { resolve } = this.initializationQueue.shift();
            resolve(result);
        }
    }
    rejectQueue(error) {
        while (this.initializationQueue.length > 0) {
            const { reject } = this.initializationQueue.shift();
            reject(error);
        }
    }
    async startExchangeProvider(provider) {
        if (await (0, utils_1.handleBanStatus)(await (0, utils_1.loadBanStatus)())) {
            return null;
        }
        if (!provider) {
            throw new Error("Provider is required to start exchange provider.");
        }
        this.exchangeProvider =
            this.exchangeCache.get(provider) ||
                (await this.initializeExchange(provider));
        return this.exchangeProvider;
    }
    removeExchange(provider) {
        if (!provider) {
            throw new Error("Provider is required to remove exchange.");
        }
        this.exchangeCache.delete(provider);
        if (this.provider === provider) {
            this.exchange = null;
            this.provider = null;
        }
    }
    async getProvider() {
        if (!this.provider) {
            this.provider = await this.fetchActiveProvider();
        }
        return this.provider;
    }
    async testExchangeCredentials(provider) {
        if (await (0, utils_1.handleBanStatus)(await (0, utils_1.loadBanStatus)())) {
            return {
                status: false,
                message: "Service temporarily unavailable. Please try again later.",
            };
        }
        try {
            const apiKey = process.env[`APP_${provider.toUpperCase()}_API_KEY`];
            const apiSecret = process.env[`APP_${provider.toUpperCase()}_API_SECRET`];
            const apiPassphrase = process.env[`APP_${provider.toUpperCase()}_API_PASSPHRASE`];
            if (!apiKey || !apiSecret || apiKey === "" || apiSecret === "") {
                return {
                    status: false,
                    message: "API credentials are missing from environment variables",
                };
            }
            // Create exchange instance with timeout and error handling
            const exchange = new ccxt.pro[provider]({
                apiKey,
                secret: apiSecret,
                password: apiPassphrase,
                timeout: 30000, // 30 second timeout
                enableRateLimit: true,
            });
            // Test connection by loading markets first
            await exchange.loadMarkets();
            // Test credentials by fetching balance
            const balance = await exchange.fetchBalance();
            // Clean up the connection
            await exchange.close();
            if (balance && typeof balance === 'object') {
                return {
                    status: true,
                    message: "API credentials are valid and connection successful",
                };
            }
            else {
                return {
                    status: false,
                    message: "Failed to fetch balance with the provided credentials",
                };
            }
        }
        catch (error) {
            (0, logger_1.logError)("exchange", error, __filename);
            // Handle specific error types
            if (error.name === 'AuthenticationError') {
                return {
                    status: false,
                    message: "Invalid API credentials. Please check your API key and secret.",
                };
            }
            else if (error.name === 'NetworkError' || error.code === 'ENOTFOUND' || error.code === 'EAI_AGAIN') {
                return {
                    status: false,
                    message: "Network error. Please check your internet connection and try again.",
                };
            }
            else if (error.name === 'ExchangeNotAvailable') {
                return {
                    status: false,
                    message: "Exchange service is temporarily unavailable. Please try again later.",
                };
            }
            else if (error.name === 'RateLimitExceeded') {
                return {
                    status: false,
                    message: "Rate limit exceeded. Please wait a moment and try again.",
                };
            }
            else if (error.name === 'PermissionDenied') {
                return {
                    status: false,
                    message: "Insufficient API permissions. Please check your API key permissions.",
                };
            }
            else {
                return {
                    status: false,
                    message: `Connection failed: ${error.message || 'Unknown error occurred'}`,
                };
            }
        }
    }
    async stopExchange() {
        if (this.exchange) {
            await this.exchange.close();
            this.exchange = null;
        }
    }
}
ExchangeManager.instance = new ExchangeManager();
exports.default = ExchangeManager.instance;
function mapChainNameToChainId(chainName) {
    const chainMap = {
        BEP20: "bsc",
        BEP2: "bnb",
        ERC20: "eth",
        TRC20: "trx",
        "KAVA EVM CO-CHAIN": "kavaevm",
        "LIGHTNING NETWORK": "lightning",
        "BTC-SEGWIT": "btc",
        "ASSET HUB(POLKADOT)": "polkadot",
    };
    return chainMap[chainName] || chainName;
}
