"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.sendEmailWithProvider = sendEmailWithProvider;
exports.emailWithSendgrid = emailWithSendgrid;
exports.emailWithNodemailerService = emailWithNodemailerService;
exports.emailWithNodemailerSmtp = emailWithNodemailerSmtp;
exports.prepareEmailTemplate = prepareEmailTemplate;
exports.fetchAndProcessEmailTemplate = fetchAndProcessEmailTemplate;
exports.replaceTemplateVariables = replaceTemplateVariables;
const mail_1 = __importDefault(require("@sendgrid/mail"));
const fs_1 = __importDefault(require("fs"));
const nodemailer_1 = __importDefault(require("nodemailer"));
const constants_1 = require("./constants");
const error_1 = require("./error");
const db_1 = require("@b/db");
const logger_1 = require("@b/utils/logger");
const emailTemplates_1 = require("./emailTemplates");
async function sendEmailWithProvider(provider, options) {
    console.log(`[EMAIL DEBUG] Starting email send with provider: ${provider}`);
    console.log(`[EMAIL DEBUG] Recipient: ${options.to}`);
    console.log(`[EMAIL DEBUG] Subject: ${options.subject}`);
    try {
        switch (provider) {
            case "local":
                const localSenderName = process.env.APP_EMAIL_SENDER_NAME || constants_1.NEXT_PUBLIC_SITE_NAME || "Support";
                const localSenderEmail = process.env.NEXT_PUBLIC_APP_EMAIL || "no-reply@localhost";
                options.from = `"${localSenderName}" <${localSenderEmail}>`;
                console.log(`[EMAIL DEBUG] Local SMTP - From: ${options.from}`);
                await emailWithLocalSMTP(options);
                break;
            case "nodemailer-service":
                const serviceSenderName = process.env.APP_EMAIL_SENDER_NAME || constants_1.NEXT_PUBLIC_SITE_NAME || "Support";
                options.from = `"${serviceSenderName}" <${constants_1.APP_NODEMAILER_SERVICE_SENDER}>`;
                console.log(`[EMAIL DEBUG] Nodemailer Service - From: ${options.from}`);
                console.log(`[EMAIL DEBUG] Service: ${constants_1.APP_NODEMAILER_SERVICE}`);
                await emailWithNodemailerService(constants_1.APP_NODEMAILER_SERVICE_SENDER, constants_1.APP_NODEMAILER_SERVICE_PASSWORD, constants_1.APP_NODEMAILER_SERVICE, options);
                break;
            case "nodemailer-smtp":
                const senderEmail = process.env.NEXT_PUBLIC_APP_EMAIL &&
                    process.env.NEXT_PUBLIC_APP_EMAIL !== ""
                    ? process.env.NEXT_PUBLIC_APP_EMAIL
                    : constants_1.APP_NODEMAILER_SMTP_SENDER;
                const senderName = process.env.APP_EMAIL_SENDER_NAME || constants_1.NEXT_PUBLIC_SITE_NAME || "Support";
                options.from = `"${senderName}" <${senderEmail}>`;
                console.log(`[EMAIL DEBUG] SMTP - From: ${options.from}`);
                console.log(`[EMAIL DEBUG] SMTP Host: ${constants_1.APP_NODEMAILER_SMTP_HOST}:${constants_1.APP_NODEMAILER_SMTP_PORT}`);
                console.log(`[EMAIL DEBUG] SMTP Encryption: ${constants_1.APP_NODEMAILER_SMTP_ENCRYPTION}`);
                // Determine secure setting based on port and encryption setting
                const isSecure = constants_1.APP_NODEMAILER_SMTP_PORT === "465" || constants_1.APP_NODEMAILER_SMTP_ENCRYPTION === "ssl";
                await emailWithNodemailerSmtp(constants_1.APP_NODEMAILER_SMTP_SENDER, constants_1.APP_NODEMAILER_SMTP_PASSWORD, constants_1.APP_NODEMAILER_SMTP_HOST, constants_1.APP_NODEMAILER_SMTP_PORT, isSecure, options);
                break;
            case "nodemailer-sendgrid":
                const sendgridSenderName = process.env.APP_EMAIL_SENDER_NAME || constants_1.NEXT_PUBLIC_SITE_NAME || "Support";
                options.from = `"${sendgridSenderName}" <${constants_1.APP_SENDGRID_SENDER}>`;
                console.log(`[EMAIL DEBUG] SendGrid - From: ${options.from}`);
                await emailWithSendgrid(options);
                break;
            default:
                console.error(`[EMAIL DEBUG] Unsupported provider: ${provider}`);
                throw new Error("Unsupported email provider");
        }
        console.log(`[EMAIL DEBUG] Email sent successfully to ${options.to}`);
    }
    catch (error) {
        console.error(`[EMAIL DEBUG] Error sending email:`, error);
        (0, logger_1.logError)("email", error, __filename);
        throw error;
    }
}
async function emailWithLocalSMTP(options) {
    try {
        const transporterOptions = {
            sendmail: true,
            newline: "unix",
            path: constants_1.APP_SENDMAIL_PATH,
        };
        const APP_NODEMAILER_DKIM_PRIVATE_KEY = process.env.APP_NODEMAILER_DKIM_PRIVATE_KEY || "";
        const APP_NODEMAILER_DKIM_DOMAIN = process.env.APP_NODEMAILER_DKIM_DOMAIN || "";
        const APP_NODEMAILER_DKIM_SELECTOR = process.env.APP_NODEMAILER_DKIM_SELECTOR || "default";
        if (APP_NODEMAILER_DKIM_PRIVATE_KEY &&
            APP_NODEMAILER_DKIM_DOMAIN &&
            APP_NODEMAILER_DKIM_SELECTOR) {
            transporterOptions.dkim = {
                privateKey: fs_1.default.readFileSync(APP_NODEMAILER_DKIM_PRIVATE_KEY, "utf8"),
                domainName: APP_NODEMAILER_DKIM_DOMAIN,
                keySelector: APP_NODEMAILER_DKIM_SELECTOR,
            };
        }
        const transporter = nodemailer_1.default.createTransport(transporterOptions);
        const mailOptions = {
            from: options.from,
            to: options.to,
            subject: options.subject,
            html: options.html,
            text: options.text,
        };
        await transporter.sendMail(mailOptions);
    }
    catch (error) {
        (0, logger_1.logError)("email", error, __filename);
        throw error;
    }
}
async function emailWithSendgrid(options) {
    const apiKey = constants_1.APP_SENDGRID_API_KEY;
    if (!apiKey)
        throw (0, error_1.createError)({
            statusCode: 500,
            message: "Sendgrid Api key not found. Cannot send email. Aborting.",
        });
    try {
        mail_1.default.setApiKey(apiKey);
        const msg = {
            to: options.to,
            from: options.from,
            subject: options.subject,
            html: options.html ? options.html : options.text,
        };
        await mail_1.default.send(msg);
    }
    catch (error) {
        (0, logger_1.logError)("email", error, __filename);
        throw error;
    }
}
async function emailWithNodemailerService(sender, password, service, options) {
    const emailOptions = {
        from: options.from,
        to: options.to,
        subject: options.subject,
        html: options.html,
    };
    console.log(`[EMAIL DEBUG] Nodemailer Service Config:`);
    console.log(`[EMAIL DEBUG] - Service: ${service}`);
    console.log(`[EMAIL DEBUG] - Sender: ${sender}`);
    console.log(`[EMAIL DEBUG] - Has Password: ${password ? 'Yes' : 'No'}`);
    if (!service)
        throw (0, error_1.createError)({
            statusCode: 500,
            message: "Email service not specified. Aborting email send.",
        });
    if (!sender)
        throw (0, error_1.createError)({
            statusCode: 500,
            message: "Email user not specified. Aborting email send.",
        });
    if (!password)
        throw (0, error_1.createError)({
            statusCode: 500,
            message: "Email password not specified. Aborting email send.",
        });
    try {
        console.log(`[EMAIL DEBUG] Creating transporter...`);
        const transporter = await nodemailer_1.default.createTransport({
            service: service,
            auth: {
                user: sender,
                pass: password,
            },
            tls: {
                rejectUnauthorized: false,
            },
        });
        console.log(`[EMAIL DEBUG] Verifying transporter...`);
        await transporter.verify();
        console.log(`[EMAIL DEBUG] Transporter verified successfully`);
        console.log(`[EMAIL DEBUG] Sending email...`);
        const info = await transporter.sendMail(emailOptions);
        console.log(`[EMAIL DEBUG] Email sent! Message ID: ${info.messageId}`);
        console.log(`[EMAIL DEBUG] Response: ${info.response}`);
    }
    catch (error) {
        console.error(`[EMAIL DEBUG] Service error:`, error);
        (0, logger_1.logError)("email", error, __filename);
        throw error;
    }
}
async function emailWithNodemailerSmtp(sender, password, host, port, smtpEncryption, options) {
    const emailOptions = {
        from: options.from,
        to: options.to,
        subject: options.subject,
        html: options.html,
    };
    console.log(`[EMAIL DEBUG] SMTP Config:`);
    console.log(`[EMAIL DEBUG] - Host: ${host}`);
    console.log(`[EMAIL DEBUG] - Port: ${port}`);
    console.log(`[EMAIL DEBUG] - Sender: ${sender}`);
    console.log(`[EMAIL DEBUG] - Has Password: ${password ? 'Yes' : 'No'}`);
    console.log(`[EMAIL DEBUG] - Encryption: ${smtpEncryption}`);
    if (!host)
        throw (0, error_1.createError)({
            statusCode: 500,
            message: "Email host not specified. Aborting email send.",
        });
    if (!sender)
        throw (0, error_1.createError)({
            statusCode: 500,
            message: "Email user not specified. Aborting email send.",
        });
    if (!password)
        throw (0, error_1.createError)({
            statusCode: 500,
            message: "Email password not specified. Aborting email send.",
        });
    try {
        console.log(`[EMAIL DEBUG] Creating SMTP transporter...`);
        const transportConfig = {
            host: host,
            port: parseInt(port),
            pool: true,
            secure: smtpEncryption,
            auth: {
                user: sender,
                pass: password,
            },
            tls: {
                rejectUnauthorized: false,
            },
            debug: true,
            logger: true
        };
        console.log(`[EMAIL DEBUG] Transport config:`, JSON.stringify(transportConfig, null, 2));
        const transporter = await nodemailer_1.default.createTransport(transportConfig);
        console.log(`[EMAIL DEBUG] Verifying SMTP connection...`);
        await transporter.verify();
        console.log(`[EMAIL DEBUG] SMTP connection verified`);
        console.log(`[EMAIL DEBUG] Sending email via SMTP...`);
        const info = await transporter.sendMail(emailOptions);
        console.log(`[EMAIL DEBUG] Email sent! Message ID: ${info.messageId}`);
        console.log(`[EMAIL DEBUG] Accepted recipients:`, info.accepted);
        console.log(`[EMAIL DEBUG] Rejected recipients:`, info.rejected);
        console.log(`[EMAIL DEBUG] Response: ${info.response}`);
    }
    catch (error) {
        console.error(`[EMAIL DEBUG] SMTP error:`, error);
        (0, logger_1.logError)("email", error, __filename);
        throw error;
    }
}
async function prepareEmailTemplate(processedTemplate, processedSubject) {
    const generalTemplate = (0, emailTemplates_1.loadEmailTemplate)("generalTemplate");
    if (!generalTemplate) {
        throw (0, error_1.createError)({
            statusCode: 500,
            message: "General email template not found",
        });
    }
    // Use direct logo links instead of settings
    const logoUrl = `${constants_1.NEXT_PUBLIC_SITE_URL}/img/logo/logo-text.webp`;
    const siteName = constants_1.NEXT_PUBLIC_SITE_NAME || "Bicrypto";
    const replacements = {
        "%SITE_URL%": constants_1.NEXT_PUBLIC_SITE_URL,
        "%SITE_NAME%": siteName,
        "%LOGO_URL%": logoUrl,
        "%HEADER%": processedSubject,
        "%MESSAGE%": processedTemplate,
        "%SUBJECT%": processedSubject,
        "%FOOTER%": siteName,
    };
    return Object.entries(replacements).reduce((acc, [key, value]) => replaceAllOccurrences(acc, key, value), generalTemplate);
}
async function fetchAndProcessEmailTemplate(specificVariables, templateName) {
    try {
        const templateRecord = await db_1.models.notificationTemplate.findOne({
            where: { name: templateName },
        });
        if (!templateRecord || !templateRecord.email || !templateRecord.emailBody)
            throw (0, error_1.createError)({
                statusCode: 404,
                message: "Email template not found or email not enabled",
            });
        const basicVariables = {
            URL: constants_1.NEXT_PUBLIC_SITE_URL,
        };
        const variables = {
            ...basicVariables,
            ...specificVariables,
        };
        const processedTemplate = replaceTemplateVariables(templateRecord.emailBody, variables);
        const processedSubject = replaceTemplateVariables(templateRecord.subject, variables);
        return { processedTemplate, processedSubject, templateRecord };
    }
    catch (error) {
        (0, logger_1.logError)("email", error, __filename);
        throw error;
    }
}
function replaceTemplateVariables(template, variables) {
    if (typeof template !== "string") {
        console.error("Template is not a string");
        return "";
    }
    return Object.entries(variables).reduce((acc, [key, value]) => {
        if (value === undefined) {
            console.warn(`Variable ${key} is undefined`);
            return acc;
        }
        return acc.replace(new RegExp(`%${key}%`, "g"), String(value));
    }, template);
}
function replaceAllOccurrences(str, search, replace) {
    if (str == null) {
        console.error("Input string is null or undefined");
        return "";
    }
    const regex = new RegExp(search, "g");
    return str.replace(regex, replace);
}
