"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RedisSingleton = void 0;
exports.default = default_1;
const ioredis_1 = require("ioredis");
class RedisSingleton {
    constructor() { }
    static getInstance() {
        if (!RedisSingleton.instance) {
            if (RedisSingleton.isConnecting) {
                // Wait for existing connection attempt
                return new Promise((resolve) => {
                    const checkConnection = () => {
                        if (RedisSingleton.instance) {
                            resolve(RedisSingleton.instance);
                        }
                        else {
                            setTimeout(checkConnection, 10);
                        }
                    };
                    checkConnection();
                });
            }
            RedisSingleton.isConnecting = true;
            try {
                RedisSingleton.instance = new ioredis_1.Redis({
                    host: process.env.REDIS_HOST || "localhost",
                    port: parseInt(process.env.REDIS_PORT || "6379"),
                    password: process.env.REDIS_PASSWORD,
                    db: parseInt(process.env.REDIS_DB || "0"),
                    maxRetriesPerRequest: 3,
                    enableReadyCheck: true,
                    connectTimeout: 5000,
                    commandTimeout: 5000,
                    lazyConnect: true,
                    family: 4,
                    keepAlive: 30000,
                });
                // Handle connection events
                RedisSingleton.instance.on("error", (error) => {
                    console.error("Redis connection error:", error);
                });
                RedisSingleton.instance.on("connect", () => {
                    console.log("Redis connected successfully");
                });
                RedisSingleton.instance.on("ready", () => {
                    console.log("Redis ready for commands");
                });
                RedisSingleton.instance.on("close", () => {
                    console.log("Redis connection closed");
                });
                RedisSingleton.instance.on("reconnecting", () => {
                    console.log("Redis reconnecting...");
                });
            }
            catch (error) {
                console.error("Failed to create Redis instance:", error);
                throw error;
            }
            finally {
                RedisSingleton.isConnecting = false;
            }
        }
        return RedisSingleton.instance;
    }
    // Add method to safely get with timeout
    static async safeGet(key, timeoutMs = 3000) {
        const redis = this.getInstance();
        return Promise.race([
            redis.get(key),
            new Promise((_, reject) => setTimeout(() => reject(new Error('Redis GET timeout')), timeoutMs))
        ]).catch((error) => {
            console.error(`Redis GET error for key ${key}:`, error);
            return null;
        });
    }
    // Add method to safely set with timeout
    static async safeSet(key, value, timeoutMs = 3000) {
        const redis = this.getInstance();
        return Promise.race([
            redis.set(key, value).then(() => true),
            new Promise((_, reject) => setTimeout(() => reject(new Error('Redis SET timeout')), timeoutMs))
        ]).catch((error) => {
            console.error(`Redis SET error for key ${key}:`, error);
            return false;
        });
    }
    // Add cleanup method
    static async cleanup() {
        if (RedisSingleton.instance) {
            try {
                await RedisSingleton.instance.quit();
            }
            catch (error) {
                console.error("Error during Redis cleanup:", error);
            }
            RedisSingleton.instance = null;
        }
    }
}
exports.RedisSingleton = RedisSingleton;
RedisSingleton.isConnecting = false;
// Export a function that returns the Redis instance
function default_1() {
    return RedisSingleton.getInstance();
}
